function data = export_peaks(results,feature,figures)
%export_peaks
%
% Syntax:  data = export_peaks(results,feature)
%
% Inputs:
%  - results: focus output
%  - feature: selects concentration feature to export
%             1 -> peak increments
%             2 -> peak areas
%             3 -> peak maximums
%
% Outputs:
%  - The output is given in a peak_data file saved in the dataset folder
%  - data.info         Peak information measures
%  - data.info_legend  Column descriptors of data.info
%  - data.intensity    Sample peak intensities
%
% Author:  Arnald Alonso - Rheumatology Research Group (VHIR)
% email:   arnald.alonso@vhir.org
% Website: http://www.urr.cat
% June 2013; Last revision: 01-June-2013

%------------- BEGIN CODE --------------
CT = 0.8;
if feature<1 || feature >3, feature = 1; end
sfeature  = {'increments','areas','maxs'};
intensity = zeros(length(results.peaksR),size(results.input.data,2));
info      = zeros(length(results.peaksR),8);
for i = 1:length(results.peaksR),
    if feature == 1, 
        intensity(i,:) = max([results.peaksR{i}.incs;zeros(size(results.peaksR{i}.incs))]);
    elseif feature == 2,
        intensity(i,:) = results.peaksR{i}.areas;
    else
        intensity(i,:) = results.peaksR{i}.maxs;
    end
    info(i,1) = results.peaksR{i}.index;
    info(i,2) = results.peaksR{i}.ppm;
    info(i,3) = results.peaksR{i}.window;
    info(i,4) = results.ppm_start(info(i,3));
    info(i,5) = results.ppm_end(info(i,3));
    info(i,6) = mean(results.peaksR{i}.corr);
    if size(results.peaksR{i}.incs,1)==2,
        results.peaksR{i}.incs = max(results.peaksR{i}.incs);
    end
    temp = corrcoef(results.peaksR{i}.incs,results.peaksR{i}.areas);
    info(i,7) = temp(1,2);
end
M = median(intensity');
p = 0:5:95;
P = prctile(M,p);
for i=1:length(p),
    info(M>=P(i),8) = p(i)/100;
end
M2  = mean(intensity);
inc = find(abs(M2-mean(M2))<3*std(M2));
C = corrcoef(intensity(:,inc)');
G = zeros(length(results.peaksR),4);
cand = zeros(length(results.peaksR),4);
corr = [0.95 0.9 0.85 0.8];
for i=1:4,
    id = 1;
    for j=1:length(results.peaksR),
        k = C(j,:) > corr(i);
        t = setdiff(G(k,i),0);
        if ~isempty(t),
            for tt=1:length(t),
                G(G(:,i)==t(tt),i) = t(1);
            end
            G(j,i) = t(1);
        else
            G(j,i) = id;
            id = id+1;
        end
    end
    g = zeros(length(results.peaksR),1);
    u = unique(G(:,i));
    n = zeros(1,length(u));
    for j=1:length(u),
        n(j)=length(find(G(:,i)==u(j)));
    end
    [~,b] = sort(n,'descend');
    
    for j=1:length(u),
        g(G(:,i)==u(b(j))) = j;
        k = find(g==j);
        [~,ic] = max(C(k,k));
        cand(k(ic(1)),i) = 1;
    end
    G(:,i) = g;
end
info = [info G];
fileID = fopen(sprintf('%s/peak_data.txt',results.name),'w');
fprintf(fileID,'ID_PEAK\tPPM\tID_WINDOW\tWINDOW_L\tWINDOW_R\tQC_SC\tQC_IvsA\tQC_IP\tG_95\tG_90\tG_85\tG_80\n');
fclose(fileID);
dlmwrite(sprintf('%s/peak_data.txt',results.name),info,'delimiter','\t','-append','precision',6);

fileID = fopen(sprintf('%s/intensity_data.txt',results.name),'w');
s = '';
for i=1:length(results.samples{1}),
    s = sprintf('%s\t%s',s,results.samples{1}{i});
end
fprintf(fileID,'ID_PEAK\tPPM\tID_WINDOW\tWINDOW_L\tWINDOW_R\tQC_SC\tQC_IvsA\tQC_IP%s\n',s);
fclose(fileID);
dlmwrite(sprintf('%s/intensity_data.txt',results.name),[info(:,1:8) intensity],'delimiter','\t','-append','precision',6);
data.info = [info cand];
data.intensity = intensity;

if figures==1,
    D    = dir(results.name);
    isub = [D(:).isdir]; %# returns logical vector
    D    = {D(isub).name}';
    if any(strcmp(D,'peak_data')) == 0,
        mkdir([results.name '/peak_data']);
    end
    for i=1:length(results.peaksR),
        hFig = figure(1); 
        clf(1); set(hFig, 'Position', [0 0 800 300]);
        [n,x] = hist(results.peaksR{i}.areas(inc),round(length(results.peaksR{i}.areas(inc))/30));
        subplot(131); plot(x,n,'r-','LineWidth',2); grid on; axis tight;
        xlabel('PEAK AREA'); ylabel('COUNTS'); title('PEAK AREA DISTRIBUTION');
        [n,x] = hist((results.peaksR{i}.incs(:,inc)),round(length(results.peaksR{i}.areas(inc))/30));
        subplot(133); plot(x,n,'r-','LineWidth',2); grid on; axis tight;
        xlabel('PEAK INCREMENTS'); ylabel('COUNTS'); title('PEAK INCREMENTS DISTRIBUTION');
        t = corrcoef(results.peaksR{i}.areas(inc),(results.peaksR{i}.incs(:,inc)));
        subplot(132); plot(results.peaksR{i}.areas(inc),(results.peaksR{i}.incs(:,inc)),'r.','MarkerSize',15);
        xlabel('PEAK AREA'); ylabel('PEAK INCREMENT'); title(sprintf('SCATTER PLOT (C=%1.2f)',t(1,2))); grid on; axis tight;
        export_fig(sprintf('%s/peak_data/intensity_%d',results.name,results.peaksR{i}.index),'-jpg');
        hFig = figure(1); 
        clf(1); set(hFig, 'Position', [0 0 800 300]);
        plot(info(:,2),C(:,i),'r.','MarkerSize',15); axis tight; ylim([-1.02 1.02]); grid on;
        set(gca,'Xdir','reverse'); xlabel('ppm'); ylabel('corrcoef');title('CORRELATION COEFFICIENTS');
        hold on; plot(info(i,2),C(i,i),'b.','MarkerSize',15);
        export_fig(sprintf('%s/peak_data/correlation_%d',results.name,results.peaksR{i}.index),'-jpg');
        j = setdiff(find(C(i,:)>CT),i);
        if ~isempty(j),
            fileID = fopen(sprintf('%s/peak_data/%d.txt',results.name,results.peaksR{i}.index),'w');
            fprintf(fileID,'CORRCOEF\tID_PEAK\tPPM\tID_WINDOW\tWINDOW_L\tWINDOW_R\tQC_SC\tQC_IvsA\tQC_IP\tG_95\tG_90\tG_85\tG_80\n');
            fclose(fileID);
            dlmwrite(sprintf('%s/peak_data/%d.txt',results.name,results.peaksR{i}.index),[C(j,i) info(j,:)],'delimiter','\t','-append','precision',6);
            for k=1:length(j),
                hFig = figure(1); 
                clf(1); set(hFig, 'Position', [0 0 600 400]);
                subplot(121); plot(results.peaksR{i}.areas(inc),results.peaksR{j(k)}.areas(inc),'r.','MarkerSize',15);
                xlabel(sprintf('PEAK %d AREA',info(i,1))); ylabel(sprintf('PEAK %d AREA',info(j(k),1)));
                t = corrcoef(results.peaksR{i}.areas(inc),results.peaksR{j(k)}.areas(inc));
                title(sprintf('AREA CORRELATION (C=%1.2f)',t(1,2))); grid on; axis tight;
                subplot(122); plot((results.peaksR{i}.incs(:,inc)),(results.peaksR{j(k)}.incs(:,inc)),'r.','MarkerSize',15);
                xlabel(sprintf('PEAK %d INCREMENTS',info(i,1))); ylabel(sprintf('PEAK %d INCREMENTS',info(j(k),1)));
                t = corrcoef((results.peaksR{i}.incs(:,inc)),(results.peaksR{j(k)}.incs(:,inc)));
                title(sprintf('INCREMENT CORRELATION (C=%1.2f)',t(1,2))); grid on; axis tight;
                export_fig(sprintf('%s/peak_data/correlation_%d_%d',results.name,results.peaksR{i}.index,results.peaksR{j(k)}.index),'-jpg');
            end
        end
    end
end