function data = focus(X,ppm,parameters,iW,iP)
%focus - Alignment, peak detection and integration
%This function performs spectral alignment using FOCUS algorithm for peak
%identification and quantification.
%
% Syntax:  data = focus(X,ppm,fparameters,fig_handles)
%
% Inputs:
%  - X: Matrix containing signals column-wise.
%  - ppm: ppm values corresponding to X rows.
%  - parameters: Focus algorithm parameters.
%  - fig_handles: Figure handles.
%
% Outputs:
%
% Author:  Arnald Alonso - Rheumatology Research Group (VHIR)
% email:   arnald.alonso@vhir.org
% Website: http://www.urr.cat
% June 2013; Last revision: 01-June-2013

%------------- BEGIN CODE --------------
%% IWSS EXTRACTION:
data.ok   = 0;
Xp        = X;
[Np0,Ns0] = size(X);
i_exclude_int = find(sum(X < parameters.align.SIT) == Np0);
i_include = setdiff(1:Ns0, i_exclude_int);
if length(i_include) < Ns0*parameters.roi.min_sample_percent,
    return;
end
X       = X(:, i_include);
X2      = iwss(X, round(parameters.align.LW/abs(ppm(1)-ppm(2))), parameters.align.DR, parameters.align.SIT, parameters.align.TWR);
[~,Ns]  = size(X2);
%% SHIFT & maxCORR matrix estimation
[maxC, ddp, C]   = optshift(X2, parameters.align.corr_GW);
exclude_corr_nan = find((sum(isnan(C)) == Ns) | (max(maxC) < parameters.align.corr_min));
include_corr_nan = sort(setdiff(1:Ns,exclude_corr_nan));
i_exclude_corr   = i_include(exclude_corr_nan);
i_include        = i_include(include_corr_nan);
if length(i_include) < Ns0*parameters.roi.min_sample_percent,
    return;
end
align.sample = zeros(1, Ns0);
align.sample(1, i_include)      = 1;
align.sample(1, i_exclude_corr) = 2;
align.sample(1, i_exclude_int)  = 3;
align.maxcorr = zeros(1, Ns0)-1;
align.maxcorr(i_include) = max(maxC(include_corr_nan, include_corr_nan));
X    = X(:, include_corr_nan);
X2   = X2(:, include_corr_nan);
maxC = maxC(include_corr_nan,include_corr_nan);
ddp  = ddp(include_corr_nan,include_corr_nan);
C    = C(include_corr_nan,include_corr_nan);
%% EXPANDING DATA MATRICES
X2       = [zeros(size(X2));X2;zeros(size(X2))];
X        = [zeros(size(X));X;zeros(size(X))];
[Np, ~]  = size(X2);
%% FIGURE
if parameters.figures(2)>0,
    corrXinit  = mean(mean(corrcoef(X((((Np+3)/3)+1):(2*((Np+3)/3)),:))));
    corrX2init = mean(mean(corrcoef(X2(((Np/3)+1):(2*(Np/3)),:))));
    figure(parameters.figures(2));
    subplot(4,2,1);plot(ppm,X((((Np+3)/3)+1):(2*((Np+3)/3)),:)); title(sprintf('Initial spectra (Mean corr = %1.3f)',corrXinit)); xlabel('ppm'); ylabel('sample'); grid on; axis tight; set(gca,'XDir','reverse');
    subplot(4,2,2);imagesc(X2(((Np/3)+1):(2*(Np/3)),:)'); colorbar(); title(sprintf('Initial INT-weighted slope signal (Mean corr = %1.3f)',corrX2init)); xlabel('n'); ylabel('sample');grid on;
    subplot(4,2,3:4);imagesc(C,[0 1]); colorbar(); title('INITIAL CORRELATION MATRIX'); xlabel('sample'); ylabel('sample');
end
%% RECURSIVE ALIGNMENT
[X,X2,dt,step] = recshift(X,X2,ddp,maxC,parameters.align.maxiter,parameters.align.corr_min);
[X,X2,dt2]     = fineshift(X,X2);
dt = [dt; dt2];
corrXend  = mean(mean(corrcoef(X((((Np+3)/3)+1):(2*((Np+3)/3)),:))));
if (parameters.figures(2)>0),
    corrX2end = mean(mean(corrcoef(X2(((Np/3)+1):(2*(Np/3)),:))));
    subplot(4,2,7);plot(ppm,X((((Np+3)/3)+1):(2*((Np+3)/3)),:)); title(sprintf('Aligned spectra (Mean corr = %1.3f)',corrXend)); xlabel('ppm'); ylabel('sample'); grid on; axis tight;set(gca,'XDir','reverse');
    subplot(4,2,8);imagesc(X2(((Np/3)+1):(2*(Np/3)),:)'); colorbar(); title(sprintf('Aligned INT-weighted slope signal (Mean corr = %1.3f)',corrX2end)); xlabel('n'); ylabel('sample');grid on;
    if ~isempty(find(sum(dt,1)~=0, 1)),
        subplot(4,2,5); ldt = log10(abs(dt)+0.1); mn = min(ldt(:)); rng = max(ldt(:))-mn; ldt = 1+63*(ldt-mn)/rng;
        image(ldt); hC = colorbar; L = 10.^linspace(-1,log10(max(abs(dt(:)))),6); l = 1+63*(log10(L)-mn)/rng; set(hC,'Ytick',l,'YTicklabel',round(L));
        xlabel('sample'); ylabel('step'); title('BY-STEP APPLIED SHIFTS');
    end
    subplot(4,2,6); bar(sum(dt)); title('GLOBAL SHIFT'); xlabel('sample'); ylabel('points');grid on; axis tight; hold on;
    plot([1 size(dt,2)],ones(1,2)*mean(sum(dt)),'r-');plot([1 size(dt,2)],ones(1,2)*mean(sum(dt))+3*std(sum(dt)),'r--');plot([1 size(dt,2)],ones(1,2)*mean(sum(dt))-3*std(sum(dt)),'r--'); axis tight;
    export_fig(sprintf('%s/figures/alignment_%d',parameters.id,iW),'-jpg');
end
align.steps  = step;
align.shift  = -1*ones(1,Ns0);
align.shift(i_include) = sum(dt,1);
align.corr_w = corrXend;
%% PEAK DETECTION
Np = size(X2,1);
[ris, f, lim_left, lim_right, center] = peak_extraction(X,ppm,parameters.peak.DFL,parameters.peak.pS,parameters.peak.pW,parameters.peak.pT,parameters.figures(3));
if parameters.figures(3)>0,
    export_fig(sprintf('%s/figures/peak_window_%d',parameters.id,iW),'-jpg');
end
align.AAS = [ris'; f'];
%% PEAK INTEGRATION
X = X((((Np+3)/3)+1):(2*((Np+3)/3)),:);
Xpp = Xp;
Xp(:,i_include) = X;
peaks = cell(1,length(lim_left));
for i = 1:length(lim_left),
    peaks{i}.window= iW;
    peaks{i}.id    = iP;
    peaks{i}.start = lim_left(i);
    peaks{i}.end   = lim_right(i);
    peaks{i}.corr = zeros(1,size(Xp,2))-2;
    ii = find( (sum(Xp(round(lim_left(i)):round(lim_right(i)),:))>0) & (std(Xp(round(lim_left(i)):round(lim_right(i)),:))>0) );
    peaks{i}.corr(ii) = mean(corrcoef(Xp(round(lim_left(i)):round(lim_right(i)),ii)));
    peaks{i}.pos  = (center(i)+align.shift)*(ppm(2)-ppm(1))+ppm(1);
    iii= find(peaks{i}.corr(ii)>0);
    peaks{i}.ppm   = sum(peaks{i}.pos(ii(iii)).*peaks{i}.corr(ii(iii))/sum(peaks{i}.corr(ii(iii))));
    peaks{i}.ratioL= ris(round(lim_left(i)))/max(ris(round(lim_left(i)):round(lim_right(i))));
    peaks{i}.ratioR= ris(round(lim_right(i)))/max(ris(round(lim_left(i)):round(lim_right(i))));
	[MAX,peaks{i}.dev] = max(Xp(round(lim_left(i)):round(lim_right(i)),:));
	MIN = min(Xp(round(lim_left(i)):round(lim_right(i)),:));
    %%peaks{i}.areas = sum(Xp(round(lim_left(i)):round(lim_right(i)),:));
	peaks{i}.areas = sum(Xp(round(lim_left(i)):round(lim_right(i)),:))-length(round(lim_left(i)):round(lim_right(i)))*MIN;
    peaks{i}.mid   = Xp(round(mean(round(lim_left(i)):round(lim_right(i)))),:);
    peaks{i}.maxs = MAX;
    %%peaks{i}.incs  = [peaks{i}.mid-Xp(round(lim_left(i)),:);peaks{i}.mid-Xp(round(lim_right(i)),:)];
	peaks{i}.incs = MAX-MIN;
    [~,dw]      = max(ris(round(lim_left(i)):round(lim_right(i))));
    peaks{i}.dev   = abs(peaks{i}.dev - dw);
    if (parameters.figures(4)>0),
        figure(parameters.figures(4)); clf(parameters.figures(4));
        subplot(4,4,1:4); 
        for j=1:size(Xpp,2), 
            jj = (peaks{i}.start:peaks{i}.end)+align.shift(j);
            ijj= find((jj>=1)&(jj<=length(ppm)));
            if ~isempty(ijj), plot(ppm(jj(ijj)),Xpp(jj(ijj),j),'k-','LineWidth',3.5); hold on; end
        end
        plot(ppm,Xpp);title(sprintf('UNALIGNED SPECTRA (Window=%d / Peak=%d)',iW,i)); grid on; set(gca,'XDir','reverse');
        subplot(4,4,5:8); plot(ppm,Xp,'LineWidth',1.3); title('ALIGNED SPECTRA');   grid on; hold on;set(gca,'XDir','reverse');
        area(ppm([round(lim_left(i)) round(lim_right(i))]),ones(1,2)*max(Xp(:))*1.1,'FaceColor','g'); alpha(0.4); axis tight;
        subplot(4,4,9:12); plot(ppm(peaks{i}.start:peaks{i}.end),Xp(peaks{i}.start:peaks{i}.end,:),'LineWidth',1.3); title('ZOOMED ALIGNED SPECTRA');   grid on; axis tight;set(gca,'XDir','reverse');
        subplot(4,4,13); plot(peaks{i}.corr,'.'); axis tight; grid on; ylim([-1 1]); xlabel('sample'); title(sprintf('SAMPLE CORRELATION')); ylabel('corr');
        y = prctile(peaks{i}.corr,25:25:75); hold on;
        for kk=1:length(y), plot([0 length(peaks{i}.corr)],[1 1]*y(kk),'r--'); end
        plot(peaks{i}.corr,'.');
        subplot(4,4,14); plot(min(peaks{i}.incs),'.'); axis tight; grid on; title(sprintf('INCREMENTS')); xlabel('sample'); ylabel('min incr');
        y = prctile(min(peaks{i}.incs),25:25:75); hold on;
        for kk=1:length(y), plot([0 length(peaks{i}.corr)],[1 1]*y(kk),'r--'); end
        plot(min(peaks{i}.incs),'.');
        t = corrcoef(peaks{i}.areas,peaks{i}.maxs);
        subplot(4,4,15); plot(peaks{i}.areas,peaks{i}.maxs,'.'); axis tight; grid on; xlabel('area');ylabel('max');title(sprintf('AREA vs MAX: QC=%1.3f',t(1,2)));
        hold on; means = [mean(peaks{i}.areas) mean(peaks{i}.maxs)]; stds = [std(peaks{i}.areas) std(peaks{i}.maxs)];
        plot([means(1) means(1)],[0 max(peaks{i}.maxs)],'r--','LineWidth',1.5);
        plot([0 max(peaks{i}.areas)],[means(2) means(2)],'r--','LineWidth',1.5);
        area([means(1)-4*stds(1) means(1)+4*stds(1)],[means(2)+4*stds(2) means(2)+4*stds(2)],means(2)-4*stds(2),'FaceColor','m');alpha(0.4);
        subplot(4,4,16); plot(align.shift,'.'); axis tight; grid on; xlabel('sample');ylabel('shift');title(sprintf('ALIGNMENT SHIFTS: STD=%1.3f',std(align.shift)));
        export_fig(sprintf('%s/figures/peak_%d',parameters.id,iP),'-jpg');
    end
    iP = iP + 1;
end
data.align   = align;
data.peaks   = peaks;
data.ok      = 1;
