function results = focus_workflow(file_data,ppm_data,sample_data,parameters,segments)
%FOCUS_WORKFLOW - Analysis Workflow for metabolomics NMR data
%This function performs alignment, peak detection, background substraction and
%peak intensity extraction over NMR spectra of multiple samples.
%
% Syntax:  results = focus_workflow(file_data,ppm_data,sample_data,parameters)
%
% Inputs:
%  - file_data:   Filename of the tab separated raw spectrum data. Samples by columns and ppm 
%                 intensities by rows.
%  - ppm_data:    Filename of the one-column file providing ppm spectrum values.
%  - sample_data: Filename of the one-column file providing sample names
%  column-wise
%  - parameters:  Structure array with the algorithm parameters (see $focus_parameters)
%  - segments:    For automated analysis: segments = []. 
%                 User-defined segments: i.e. segments = [[1.23 1.25];[7.50 7.65]];
%
% Outputs:
%  A folder named as the analysis descriptor is created to save all the
%  data related with the analysis.
%  - results: Structure array with the following fields:
%       name: Analysis descriptor
%       parameters: Algorithm parameters used
%       input.data: Input spectral intensity data
%       input.ppm: Input spectral ppm points
%       N_start: Windows starting points
%       N_end: Windows ending points
%       ppm_start: Windows starting ppms
%       ppm_end:  Windows ending ppms
%       processed: Processed windows (1) and non-processed windows (0)
%       windows: Processed windows information
%           - sample   Samples analyzed within the corresponding window
%           - maxcorr  Per-sample maximum correlation over the window
%           - steps    Number of recursive steps applied until alignment convergence
%           - shift    Correction shift applied to each sample
%           - corr_w   Average window correlation
%           - AAS      Peak detection information signals
%       peaks_info: Window id where each peak has been detected
%       peaks: Peak information
%           - window   Window id
%           - id       Peak id
%           - start    Starting point
%           - end      Ending point
%           - corr     Per-sample peak shape correlation
%           - pos      Per-sample peak center
%           - ppm      Peak position
%           - ratioL   Maximum / left_side intensity ratio
%           - ratioR   Maximum / right_side intensity ratio
%           - areas    Per-sample peak areas
%           - mid      Per-sample median window value
%           - maxs     Per-sample maximum peak intensities
%           - dev      Per-sample shifts
%           - incs     Per-sample increments
%       peaksR: Peak information of the reduced peak dataset
%
% See also: focus_parameters, compound_identification

% Author:  Arnald Alonso - Rheumatology Research Group (VHIR)
% email:   arnald.alonso@vhir.org
% Website: http://www.urr.cat
% June 2013; Last revision: 01-June-2013
%------------- BEGIN CODE --------------
fprintf(' --------------------------------------\n');
fprintf('|   FOCUS NMR DATA WORKFLOW ANALYSIS   |\n');
fprintf(' --------------------------------------\n');
fprintf('ANALYSIS STARTED AT %s\n',datestr(now));
tstart = cputime;
%% USER-SEGMENT ANALYSIS
%% CREATES DIRECTORY STRUCTURE
D    = dir();
isub = [D(:).isdir]; %# returns logical vector
D    = {D(isub).name}';
if any(strcmp(D,parameters.id)) == 1,
    D = dir(parameters.id);
    isub = [D(:).isdir]; %# returns logical vector
    D = {D(isub).name}';
    if any(strcmp(D,'figures'))==0,
        mkdir(parameters.id,'figures');
    end
else
    mkdir(parameters.id);
    mkdir(parameters.id,'figures');
end
%% LOADS RAW INPUT DATA
fprintf('- Reading input data...\n');
data = dlmread(file_data,'\t');
ppm  = dlmread(ppm_data,'\t');
file = fopen(sample_data,'r');
samples = textscan(file,'%s',size(data,2));
fclose(file);
%% INFORMATIVE SPECTRAL SEGMENTS
if isempty(segments),
    fprintf('- Computing global spectrum and non-zero segments...\n');
    gsp     = sum(data,2);
    gspb    = sum((data > parameters.roi.it),2) / size(data,2);
    inz     = gspb > parameters.roi.min_sample_percent; 
    nzsegm  = zeros(1,length(ppm));
    nzsegm(inz) = 1;
    LW_open = round(parameters.roi.min_width / abs(ppm(2)-ppm(1)));
    SE1     = strel('line', LW_open,   0);
    SE2     = strel('line', LW_open*2, 0);
    nzsegm  = imdilate(imerode(nzsegm, SE1), SE2);
    %% FIGURE 1
    if parameters.figures(1) > 0, 
        p98  = prctile(gsp,99);
        hFig = figure(parameters.figures(1)); 
        clf(parameters.figures(1)); set(hFig, 'Position', [0 0 1800 400]);
        area(ppm, p98*abs(1-nzsegm), 'FaceColor', 'm', 'EdgeColor', 'm');
        hold on; area(ppm, p98*abs(nzsegm), 'FaceColor', 'c', 'EdgeColor', 'c');
        plot(ppm, gsp, 'b-'); grid(); axis tight; title('GLOBAL SPECTRUM');
        xlabel('chemical shift (ppm)'); ylabel('intensity'); ylim([0 p98]); grid on;
        legend('non-informative segment', 'informative segment', 'global spectrum'); set(gca,'XDir','reverse');
        export_fig(sprintf('%s/informative_spectral_segments',parameters.id),'-jpg');
    end
else
    nzsegm = ones(1,length(ppm));
end
%% DOWNSAMPLING AND WINDOW SAMPLE LENGTH CALCULATION
Xtemp    = data(1:parameters.window.fs:size(data,1),:);
nztemp   = nzsegm(1:parameters.window.fs:length(nzsegm));
LWs_ppm  = parameters.window.length;
LWs_N    = round(parameters.window.length/abs(ppm(1)-ppm(2)));
LWs_N    = round(LWs_N/parameters.window.fs);
LWs_N    = LWs_N + mod(LWs_N,2);
ppm      = ppm(1:parameters.window.fs:end);
%% SLIDING WINDOW ANALYSIS
LW      = LWs_N; 
if isempty(segments),
    Np      = size(Xtemp,1);
    N_start = 1:(LW*(1-parameters.window.ovlap)):Np;
    N_end   = N_start + LW -1;
    N_start = N_start(N_end <= Np);
    N_end   = N_end(N_end <= Np);
else
    N = zeros(size(segments,1),2);
    for i=1:size(segments,1),
        [~,N(i,2)] = min(abs(ppm-segments(i,1)));
        [~,N(i,1)] = min(abs(ppm-segments(i,2)));
    end
    N_start = N(:,1);
    N_end = N(:,2);
end
Nw      = length(N_end);
results.name = parameters.id;
results.parameters = parameters;
results.input.data = data;
results.input.ppm  = ppm;
results.N_start   = N_start;
results.N_end     = N_end;
results.ppm_start = ppm(N_start)';
results.ppm_end   = ppm(N_end)';
fprintf('- Sliding window analysis: LengthWindow = %d points / NumberWindows = %d\n',LW,Nw);
windows    = cell(1,length(N_end));
peaks      = cell(1,(length(N_end)*5));
peaks_info = zeros(1,5*length(N_end));
processed  = zeros(1,length(N_end));
kk  = 1;
for j=1:length(N_end),
    tis = cputime;
    tot = sum(nztemp(N_start(j):N_end(j)));
    if (tot/LW) >= parameters.roi.min_segment_percent,
        % RESET FIGURES
        for jj=2:length(parameters.figures),
            if (parameters.figures(jj)>0), 
                hFig = figure(parameters.figures(jj)); 
                clf(parameters.figures(jj)); set(hFig, 'Position', [0 0 2000 2200]);
                figure(parameters.figures(jj)); clf(parameters.figures(jj)); 
            end
        end
        % FOCUS WINDOW ANALYSIS
        results_focus.ok = 0;
        results_focus  = focus(Xtemp(N_start(j):N_end(j),:),ppm([N_start(j):N_end(j)]),parameters,j,kk);
        if results_focus.ok == 1, 
            processed(1,j)= 1; 
            windows{j}    = results_focus.align;
            for k=1:length(results_focus.peaks), 
                peaks{kk}  = results_focus.peaks{k}; 
                peaks_info(1,kk)= j;
                kk=kk+1; 
            end
        end
        fprintf('    LW=%d\ti=%d/%d\tprocessing_time=%1.2fmin\n',LW,j,length(N_start),(cputime-tis)/60);
    end
end
results.processed = processed;
results.windows = windows;
results.peaks_info = peaks_info(1,1:(kk-1));
results.peaks   = peaks(1:(kk-1));
fprintf('  Sliding window analysis end: time=%1.2fmin\n',(cputime-tis)/60);
fprintf('- Applying peak reduction...');
results = peak_reduce(results,parameters.pr,1);
results.samples = samples;
save(sprintf('%s/results.mat',results.name), 'results');
fprintf('ANALYSIS FINISHED AT %s\n',datestr(now));
%------------- END OF CODE --------------