function [X, X2, dt, step] = recshift(X, X2, ddp, maxC, MAX_step, min_corr)
%recshift - Recursive alignment
%This function performs iterative alignment of spectral signals based on the 
%maximal correlation matrix and the optimal distance matrix.
%
% Syntax:  [X, X2, dt, step] = recshift(X, X2, ddp, maxC, MAX_step, min_corr)
%
% Inputs:
%  - X:    Matrix containing signals column-wise.
%  - X2:   Matrix containing transformed signals column-wise.
%  - ddp:  Optimal distance matrix.
%  - maxC: Maximal correlation matrix.
%  - MAX_step: Maximum number of iterations.
%  - min_corr: Minimum correlation threshold.
%
% Outputs:
%  - X: Matrix containing aligned signals column-wise.
%  - X2: Matrix containing aligned transformed signals column-wise.
%  - dt: Shifts applied to each sample in each iteration
%  - step: Number of steps required for convergence.
%
% Author:  Arnald Alonso - Rheumatology Research Group (VHIR)
% email:   arnald.alonso@vhir.org
% Website: http://www.urr.cat
% June 2013; Last revision: 01-June-2013

%------------- BEGIN CODE --------------
[Np,Ns] = size(X);
d       = zeros(1,Ns);
dt      = zeros(MAX_step,Ns);
maxC    = maxC - eye(size(maxC,1));
kstep   = 0;
while (1),
    kstep = kstep+1;
    d(:)  = 0;
    % Compute optimal shifts
    for i = 1:Ns, 
        k = find(maxC(i,:) > min_corr);
        if ~isempty(k),
            tddp = round(0.5*sum((ddp(i,k).*maxC(i,k)) / sum(maxC(i,k))));
            d(i) = tddp;
        end
    end
    dt(kstep,:) = d;
    % Apply shifts
    for i = 1:Ns,
        if d(i) > 0,     
            X2(1:(Np-3-d(i)), i)     = X2((d(i)+1):end, i);
            X2((Np-3-d(i)+1):end, i) = 0;
            X(1:(Np-d(i)), i)        = X((d(i)+1):end, i);
            X((Np-d(i)+1):end, i)    = 0;
        elseif d(i)<0, 
            X2((-d(i)+1):end, i) = X2(1:(end+d(i)), i);
            X2(1:(-d(i)), i)     = 0;
            X((-d(i)+1):end, i)  = X(1:(end+d(i)), i);
            X(1:(-d(i)), i)      = 0;
        end
    end
    % Update SHIFT MATRIX
    ddp = ddp - (d'*ones(1,Ns)) + (ones(Ns,1)*d);
    if mean(abs(dt(kstep,:)))==0, 
        dt = dt(1:kstep, :);
        break;
    end
    if kstep == MAX_step, break; end
end
step = kstep;